import io, csv, json, zipfile, random, datetime
from django.shortcuts import render
from django.http import HttpResponse
from django import forms
import pycountry
from faker import Faker

# Mapping of ISO alpha_2 to Faker locales where applicable.
LOCALE_MAP = {
    'IN': 'en_IN', 'US': 'en_US', 'GB': 'en_GB', 'CA': 'en_CA', 'AU': 'en_AU',
    'DE': 'de_DE', 'FR': 'fr_FR', 'ES': 'es_ES', 'IT': 'it_IT', 'JP': 'ja_JP',
    'CN': 'zh_CN', 'BR': 'pt_BR', 'RU': 'ru_RU', 'KR': 'ko_KR', 'MX': 'es_MX',
    'NL': 'nl_NL', 'SE': 'sv_SE', 'NO': 'nb_NO', 'FI': 'fi_FI', 'DK': 'da_DK',
    'PL': 'pl_PL', 'PT': 'pt_PT', 'TR': 'tr_TR', 'AR': 'es_AR', 'BE': 'fr_BE',
    'CH': 'de_CH', 'AT': 'de_AT', 'IE': 'en_IE', 'NZ': 'en_NZ', 'ZA': 'en_ZA',
    # add more mappings as desired; fallback used otherwise
}

# A small calling code map for many common countries. If a country is missing, phone may not have prefix.
CALLING_CODE = {
    'IN': '91','US': '1','GB': '44','CA': '1','AU': '61','DE': '49','FR': '33','ES': '34','IT': '39','JP': '81',
    'CN': '86','BR': '55','RU': '7','KR': '82','MX': '52','NL': '31','SE': '46','NO': '47','FI': '358','DK': '45',
    'PL': '48','PT': '351','TR': '90','AR': '54','BE': '32','CH': '41','AT': '43','IE': '353','NZ': '64','ZA': '27',
    # more can be added over time
}

def get_all_countries():
    countries = []
    for c in pycountry.countries:
        countries.append({'name': c.name, 'code': c.alpha_2})
    # sort by name
    countries.sort(key=lambda x: x['name'])
    return countries

class GenerateForm(forms.Form):
    count = forms.IntegerField(min_value=1, max_value=50000, initial=100)
    country_code = forms.ChoiceField(choices=[(c['code'], c['name']) for c in get_all_countries()])
    hash_passwords = forms.BooleanField(required=False)

def index(request):
    form = GenerateForm()
    return render(request, 'generator/index.html', {'form': form})

def fake_for_country(alpha2):
    locale = LOCALE_MAP.get(alpha2, 'en_US')
    try:
        f = Faker(locale)
    except Exception:
        f = Faker('en_US')
    return f

def generate_records(count, alpha2, hash_passwords=False):
    f = fake_for_country(alpha2)
    records = []
    first_names = [f.first_name() for _ in range(50)]
    last_names = [f.last_name() for _ in range(50)]
    companies = [f.company() for _ in range(20)]
    jobs = [f.job() if hasattr(f, 'job') else 'Employee' for _ in range(20)]
    for i in range(count):
        first = random.choice(first_names)
        last = random.choice(last_names)
        username = (first[0] + last + str(random.randint(1,999))).lower()
        email = f'{first.lower()}.{last.lower()}{random.randint(1,999)}@example.com'
        pwd = f.password(length=12)
        password_field = pwd  # hashing optional via CLI only in this simple UI
        address = f.address().replace('\n', ', ')
        city = f.city()
        state = getattr(f, 'state', lambda: '')()
        postal = getattr(f, 'postcode', lambda: '')()
        code = alpha2.upper()
        calling = CALLING_CODE.get(code)
        phone_local = getattr(f, 'phone_number', lambda: '0000')()
        phone = phone_local
        if calling:
            # normalize digits from phone_local, keep last n digits
            digits = ''.join([c for c in phone_local if c.isdigit()])
            if len(digits) >= 6:
                phone = f'+{calling}-{digits[-10:]}'
            else:
                phone = f'+{calling}-{digits}'
        dob = f.date_of_birth(minimum_age=18, maximum_age=75).isoformat() if hasattr(f, 'date_of_birth') else ''
        company = random.choice(companies)
        job_title = random.choice(jobs)
        bio = f'{first} {last} works as a {job_title} at {company}.'
        rec = {
            'id': i+1,
            'first_name': first,
            'last_name': last,
            'username': username,
            'email': email,
            'password': password_field,
            'address': address,
            'city': city,
            'state': state,
            'country': code,
            'postal_code': postal,
            'phone': phone,
            'date_of_birth': dob,
            'company': company,
            'job_title': job_title,
            'bio': bio,
            'is_staff': random.choice([True, False, False, False]),
            'is_superuser': False,
            'is_active': True,
            'last_login': datetime.date.today().isoformat(),
            'signup_date': datetime.date.today().isoformat(),
        }
        records.append(rec)
    return records

from django.views.decorators.http import require_POST
@require_POST
def generate_view(request):
    form = GenerateForm(request.POST)
    if not form.is_valid():
        return render(request, 'generator/index.html', {'form': form})
    count = form.cleaned_data['count']
    alpha2 = form.cleaned_data['country_code']
    hash_passwords = form.cleaned_data['hash_passwords']
    records = generate_records(count, alpha2, hash_passwords)
    # prepare files in memory
    txt_io = io.StringIO()
    for r in records:
        txt_io.write(json.dumps(r, ensure_ascii=False) + '\n')
    txt_io.seek(0)
    csv_io = io.StringIO()
    writer = csv.DictWriter(csv_io, fieldnames=list(records[0].keys()))
    writer.writeheader()
    for r in records:
        writer.writerow(r)
    csv_io.seek(0)
    zip_io = io.BytesIO()
    zf = zipfile.ZipFile(zip_io, mode='w', compression=zipfile.ZIP_DEFLATED)
    zf.writestr('users.txt', txt_io.getvalue())
    zf.writestr('users.csv', csv_io.getvalue())
    zf.writestr('manifest.json', json.dumps({'num_records': count, 'fields': list(records[0].keys()), 'country': alpha2}, indent=2))
    zf.close()
    zip_io.seek(0)
    filename = f'dummy_users_{alpha2}_{count}.zip'
    response = HttpResponse(zip_io.read(), content_type='application/zip')
    response['Content-Disposition'] = f'attachment; filename="{filename}"'
    return response
